﻿#include "precompiled.h"
#include "SceneImporter.h"
#include "common.h"

#include "Scene.h"
#include "Entity.h"
#include "Mesh.h"
#include "MeshRenderer.h"
#include "Material.h"
#include "Transform.h"
#include "ResourceManager.h"

using namespace DirectX;

//#define TraceFbx

namespace RTCam {

// Some helper functions

inline DirectX::XMFLOAT3 FbxVectorToFloat3(FbxVector4 fbxVec) {
	return DirectX::XMFLOAT3(
		static_cast<float>(fbxVec.mData[0]),
		static_cast<float>(fbxVec.mData[1]),
		static_cast<float>(fbxVec.mData[2])
	);									  
}

inline DirectX::XMFLOAT3A FbxVectorToFloat3A(FbxVector4 fbxVec) {
	return DirectX::XMFLOAT3A(
		static_cast<float>(fbxVec.mData[0]),
		static_cast<float>(fbxVec.mData[1]),
		static_cast<float>(fbxVec.mData[2])
		);									  
}

inline DirectX::XMFLOAT4 FbxVectorToFloat4(FbxVector4 fbxVec) {
	return DirectX::XMFLOAT4(
		static_cast<float>(fbxVec.mData[0]),
		static_cast<float>(fbxVec.mData[1]),
		static_cast<float>(fbxVec.mData[2]),
		static_cast<float>(fbxVec.mData[3])
	);
}


// Class functions

SceneImporter::SceneImporter(void) :
	FileImporter(),
	m_sdkManager(nullptr),
	m_geometryConverter(nullptr),
	m_resources()
{
	// Initialize the sdk manager. This object handles all our memory management.
	m_sdkManager = FbxPointer<FbxManager>(FbxManager::Create());
	m_geometryConverter = unique_ptr<FbxGeometryConverter>(new FbxGeometryConverter(m_sdkManager.get()));
}

SceneImporter::~SceneImporter(void)
{
}

void SceneImporter::Initialize(const shared_ptr<ResourceManager>& resources)
{
	m_resources = resources;
}

shared_ptr<Scene> SceneImporter::OpenScene(_In_ ID3D11Device1* device)
{
	shared_ptr<Scene> loadedScene(nullptr);

	// The file type filters
	COMDLG_FILTERSPEC fileTypes[] =
	{ 
		{ L"Autodesk FBX", L"*.fbx" },
		{ L"All files", L"*.*" },
	};

	string filepath = GetFilePath(fileTypes, ARRAYSIZE(fileTypes));
	if(filepath.length() > 0) {
		loadedScene = Import(device, filepath.c_str());
	}

	return loadedScene;
}

#pragma region DebugPrinting
//--------------------------------------------------------------------------------------
// Debug printing of imported scenes
//-------------------------------------------------------------------------------------- 
void SceneImporter::PrintTabs(int numTabs) {
    for(int i = 0; i < numTabs; ++i)
        DebugPrint("\t");
}


FbxString SceneImporter::GetAttributeTypeName(FbxNodeAttribute::EType type) {
	switch(type) {
	case FbxNodeAttribute::eUnknown: return "unidentified";
	case FbxNodeAttribute::eNull: return "null";
	case FbxNodeAttribute::eMarker: return "marker";
	case FbxNodeAttribute::eSkeleton: return "skeleton";
	case FbxNodeAttribute::eMesh: return "mesh";
	case FbxNodeAttribute::eNurbs: return "nurb";
	case FbxNodeAttribute::ePatch: return "patch";
	case FbxNodeAttribute::eCamera: return "camera";
	case FbxNodeAttribute::eCameraStereo:    return "stereo";
	case FbxNodeAttribute::eCameraSwitcher: return "camera switcher";
	case FbxNodeAttribute::eLight: return "light";
	case FbxNodeAttribute::eOpticalReference: return "optical reference";
	case FbxNodeAttribute::eOpticalMarker: return "marker";
	case FbxNodeAttribute::eNurbsCurve: return "nurbs curve";
	case FbxNodeAttribute::eTrimNurbsSurface: return "trim nurbs surface";
	case FbxNodeAttribute::eBoundary: return "boundary";
	case FbxNodeAttribute::eNurbsSurface: return "nurbs surface";
	case FbxNodeAttribute::eShape: return "shape";
	case FbxNodeAttribute::eLODGroup: return "lodgroup";
	case FbxNodeAttribute::eSubDiv: return "subdiv";
	case FbxNodeAttribute::eCachedEffect: return "cached effect";
	case FbxNodeAttribute::eLine: return "line";
	default: return "unimplemented";
	}
}

void SceneImporter::PrintAttribute(FbxNodeAttribute* attribute, int numTabs) {
	if(!attribute) return;

	FbxString typeName = GetAttributeTypeName(attribute->GetAttributeType());
	FbxString attrName = attribute->GetName();
	PrintTabs(numTabs);
	// Note: to retrieve the chararcter array of a FbxString, use its Buffer() method.
	DebugPrint("<attribute type='%hs' name='%hs'/>\n", typeName.Buffer(), attrName.Buffer());
}


void SceneImporter::PrintNode(FbxNode* node, int numTabs) {
	PrintTabs(numTabs);

	const char* nodeName = node->GetName();
	FbxDouble3 translation = node->LclTranslation.Get();
	FbxDouble3 rotation = node->LclRotation.Get();
	FbxDouble3 scaling = node->LclScaling.Get();

	// Print the contents of the node.
	DebugPrint("<node name='%hs' translation='(%f, %f, %f)' rotation='(%f, %f, %f)' scaling='(%f, %f, %f)'>\n", 
		nodeName, 
		translation[0], translation[1], translation[2],
		rotation[0], rotation[1], rotation[2],
		scaling[0], scaling[1], scaling[2]);
	numTabs++;

	// Print the node's attributes.
	for(int i = 0; i < node->GetNodeAttributeCount(); ++i)
		PrintAttribute(node->GetNodeAttributeByIndex(i), numTabs);

	// Recursively print the children nodes.
	for(int j = 0; j < node->GetChildCount(); ++j)
		PrintNode(node->GetChild(j), numTabs);

	numTabs--;
	PrintTabs(numTabs);
	DebugPrint("</node>\n");
}
#pragma endregion

shared_ptr<class Scene> SceneImporter::Import(_In_ ID3D11Device1* device, _In_z_ const char* filename)
{
	// Create the IO settings object.
	auto ioSettings = FbxPointer<FbxIOSettings>(FbxIOSettings::Create(m_sdkManager.get(), IOSROOT));
	m_sdkManager->SetIOSettings(ioSettings.get());
	
	// Create an importer using our sdk manager.
	auto importer = FbxPointer<FbxImporter>(FbxImporter::Create(m_sdkManager.get(), "importer"));

	// Use the first argument as the filename for the importer.
	if(!importer->Initialize(filename, -1, m_sdkManager->GetIOSettings())) {
		PrintMsg(L"Call to FbxImporter::Initialize() failed.\n");
		// TODO: Find the new method to be called for getting error strings from FBX SDK.
		//DebugPrint("Error returned: %s\n\n", ios->GetLastErrorString());
		return nullptr;
	}

	// Create a new fbx scene so it can be populated by the imported file.
	auto fbxScene = FbxPointer<FbxScene>(FbxScene::Create(m_sdkManager.get(),"importedScene"));

	// Import the contents of the file into the fbx scene.
	DebugPrint("Importing the scene in %s\n", filename);
	importer->Import(fbxScene.get());
	DebugPrint("Finished importing scene, now converting.\n");

	// Convert from FBX's right handed, Y-Up axis system to our left handed, Y-up axis system
	FbxAxisSystem::DirectX.ConvertScene(fbxScene.get());

	auto resources = m_resources.lock();
	ASSERT(resources != nullptr);
	
	// Create the scene that will be populated by the imported fbx scene.
	shared_ptr<Scene> scene = shared_ptr<Scene>(new Scene());
	scene->Initialize(resources);

	// Import the contents of the fbx scene into our scene, node by node
	FbxNode* rootNode = fbxScene->GetRootNode();
	if(rootNode != nullptr) {
		for(int i = 0; i < rootNode->GetChildCount(); ++i) {
			auto node = rootNode->GetChild(i);

#ifdef TraceFbx
			PrintNode(node, 0);
#endif
			ProcessNode(device, node, scene);
		}
	}

	// If no camera node was defined, add a default camera.
	if(scene->m_activeCam.lock() == nullptr) {
		const float initialCamPos[] = { 0.0f, 1.0f, -20.0f };

		auto camera = scene->AddCamera();
		camera->GetTransform()->SetLocalPosition(XMFLOAT3A(initialCamPos));
	}

	DebugPrint("Scene import process complete.\n");

	return scene;
}

void SceneImporter::ProcessNode(_In_ ID3D11Device1* device, _In_ FbxNode* node, const shared_ptr<Scene>& scene)
{
	// NOTE: Currently untested with what happens when a node has more than one attribute.

	// TODO: Handle node hierarchy

	const char* nodeName = node->GetName();
	
	// Make sure any meshes are triangulated
	// (If this is done later, pointers to attributes become invalid and have to be fetched again)
	for(int i = 0; i < node->GetNodeAttributeCount(); ++i) {
		auto attribute = node->GetNodeAttributeByIndex(i);

		if(attribute->GetAttributeType() == FbxNodeAttribute::eMesh) {
			FbxMesh* mesh = (FbxMesh*)(attribute);
			if(!mesh->IsTriangleMesh()) {
				bool success = m_geometryConverter->TriangulateInPlace(node);
				if(success) {
					DebugPrint("WARNING: Mesh in node '%s' had to be triangulated.\n", nodeName);
				} else {
					DebugPrint("ERROR: Failed to triangulate mesh in node '%s'.\n", nodeName);
				}
				break;
			}
		}
	}

	// Create the entity to represent the node
	shared_ptr<Entity> entity = Entity::Create(nodeName);
	
	FbxDouble3 translation = node->LclTranslation.Get();
	FbxDouble3 rotation = node->LclRotation.Get();
	FbxDouble3 scaling = node->LclScaling.Get();

	// Set position
	entity->GetTransform()->SetLocalPosition(FbxVectorToFloat3A(translation));

	XMFLOAT3 xyz = FbxVectorToFloat3(rotation);

	// Convert rotation from XYZ to quaternion
	// Pitch = X axis, Yaw = Y axis, Roll = Z axis
	// DEBUG: We probably can't use XMQuaternionRotationRollPitchYaw because FBX gives rotations in XYZ order,
	// and the RollPitchYaw function applies the rotations in a ZXY order like in BVH files.
	XMMATRIX xRot = XMMatrixRotationX(XMConvertToRadians(xyz.x));
	XMMATRIX yRot = XMMatrixRotationY(XMConvertToRadians(xyz.y));
	XMMATRIX zRot = XMMatrixRotationZ(XMConvertToRadians(xyz.z));
	XMMATRIX xyzRot = XMMatrixMultiply(XMMatrixMultiply(xRot, yRot), zRot);
	XMVECTOR xyzQuat = XMQuaternionNormalize(XMQuaternionRotationMatrix(xyzRot));
	XMFLOAT4A quaternion;
	XMStoreFloat4A(&quaternion, xyzQuat);
	entity->GetTransform()->SetLocalRotation(quaternion);
	
	// Set scaling
	entity->GetTransform()->SetLocalScale(FbxVectorToFloat3A(scaling));

	// Process the node's attributes.
	for(int i = 0; i < node->GetNodeAttributeCount(); ++i)
		ProcessAttrib(device, node->GetNodeAttributeByIndex(i), entity);

	// Add the entity to the scene
	scene->AddEntity(entity);

	// Recursively process the children nodes.
	for(int j = 0; j < node->GetChildCount(); ++j)
		ProcessNode(device, node->GetChild(j), scene);
}

void SceneImporter::ProcessAttrib(_In_ ID3D11Device1* device, _In_ FbxNodeAttribute* attribute, const shared_ptr<Entity>& entity)
{
	auto type = attribute->GetAttributeType();
	switch(type) {
	case FbxNodeAttribute::eMesh:
		{
			FbxMesh* mesh = (FbxMesh*)(attribute);
			ProcessMesh(device, mesh, entity);
			break;
		}
	case FbxNodeAttribute::eUnknown:
	case FbxNodeAttribute::eNull:
	case FbxNodeAttribute::eMarker:
	case FbxNodeAttribute::eSkeleton:
	case FbxNodeAttribute::eNurbs:
	case FbxNodeAttribute::ePatch:
	case FbxNodeAttribute::eCamera:
	case FbxNodeAttribute::eCameraStereo:
	case FbxNodeAttribute::eCameraSwitcher:
	case FbxNodeAttribute::eLight:
	case FbxNodeAttribute::eOpticalReference:
	case FbxNodeAttribute::eOpticalMarker:
	case FbxNodeAttribute::eNurbsCurve:
	case FbxNodeAttribute::eTrimNurbsSurface:
	case FbxNodeAttribute::eBoundary:
	case FbxNodeAttribute::eNurbsSurface:
	case FbxNodeAttribute::eShape:
	case FbxNodeAttribute::eLODGroup:
	case FbxNodeAttribute::eSubDiv:
	case FbxNodeAttribute::eCachedEffect:
	case FbxNodeAttribute::eLine:
	default:
		DebugPrint("Unhandled Fbx Node Attribute %s\n", GetAttributeTypeName(type));
		break;
	}
}

void SceneImporter::ProcessMesh(_In_ ID3D11Device1* device, _In_ FbxMesh* fbxMesh, const shared_ptr<Entity>& entity)
{
	ASSERT(fbxMesh != nullptr);
	
	// Make sure there are no degenerate polygons
	int numBadPolygons = fbxMesh->RemoveBadPolygons();
	if(numBadPolygons != 0) {
		DebugPrint("WARNING: %i bad polygons removed from mesh\n", numBadPolygons);
	}

	int numVertices = fbxMesh->GetControlPointsCount();
	int numPolygons = fbxMesh->GetPolygonCount();
	
	// TODO: Find out how/when to use these
	int numPolygonGroups = fbxMesh->GetElementPolygonGroupCount();
	// END TODO
	
	// TODO: Optimize to reuse vertices using the index array

	vector<VertexPositionNormalColor> vertices;
	vertices.reserve(fbxMesh->GetPolygonVertexCount()); // This is the maximum number of possible vertices to be displayed

	vector<uint16_t> indices;
	indices.reserve(numPolygons * 3);

	// Used for FbxGeometryElement::eByPolygonVertex mapping mode, and incremented
	// with each additional vertex being processed
	int totalVertexId = 0; 
	
	// Process each polygon
	for(int polygonId = 0; polygonId < numPolygons; ++polygonId) {
		// Make sure each polygon is a triangle
		int numVerticesPerPolygon = fbxMesh->GetPolygonSize(polygonId);
		ASSERT_MSG(numVerticesPerPolygon == 3, "Meshes need to be triangulated.");

		// Visit each vertex
		for(int polyVertexId = 0; polyVertexId < numVerticesPerPolygon; ++polyVertexId) {
			ProcessVertex(fbxMesh, polygonId, polyVertexId, totalVertexId, vertices, indices);
			++totalVertexId;
		}
		
	}

	auto nodeName = fbxMesh->GetNode()->GetName();
	auto mesh = shared_ptr<Mesh>(new Mesh(nodeName));
	mesh->Initialize(device, &vertices[0], vertices.size(), &indices[0], indices.size());

	// TODO: Add the mesh to the resource manager

	// Add the mesh to the entity
	entity->AddMeshRenderer(mesh);
}

// NOTE: This function is loosely based on the sample code in the FBX SDK
void SceneImporter::ProcessVertex(_In_ FbxMesh* fbxMesh, int polygonId, int polygonVertexId, int totalVertexId,
								  vector<VertexPositionNormalColor>& vertices, vector<uint16_t>& indices)
{
	// TODO: Check whether each vertex ever gets more than one elementVertex, because we can't handle that yet

	int fbxVertexIndex = fbxMesh->GetPolygonVertex(polygonId, polygonVertexId);
	FbxVector4* fbxVertices = fbxMesh->GetControlPoints();
	FbxVector4 fbxVertex = fbxVertices[fbxVertexIndex];

	// Retrieve the vertex position
	VertexPositionNormalColor vertex;
	vertex.position = FbxVectorToFloat3(fbxVertex.mData);
	vertex.color = DirectX::XMFLOAT4(1, 1, 1, 1);
	vertex.normal = DirectX::XMFLOAT3(0, 1, 0);

	// TODO: Check whether Element_____Count actually means each vertex can have multiple colors/normals/etc.
	int numColorsPerVertex = fbxMesh->GetElementVertexColorCount();
	int numNormalsPerVertex = fbxMesh->GetElementNormalCount();

	// Retrieve the vertex color
	for(int colorIndex = 0; colorIndex < numColorsPerVertex; ++colorIndex) {
		FbxGeometryElementVertexColor* elementColor = fbxMesh->GetElementVertexColor(colorIndex);
		int index;
		int directIndex;

		switch(elementColor->GetMappingMode()) {
		case FbxGeometryElement::eByControlPoint:
			index = fbxVertexIndex;
			break;
		case FbxGeometryElement::eByPolygonVertex:
			index = totalVertexId;
			break;
		default:
			BreakAndThrow("Unhandled VertexColor mapping mode in SceneImporter::ProcessVertex()\n");
			break;
		}

		switch(elementColor->GetReferenceMode()) {
		case FbxGeometryElement::eDirect:
			directIndex = index;
			break;
		case FbxGeometryElement::eIndexToDirect:
			directIndex = elementColor->GetIndexArray().GetAt(index);
			break;
		default:
			BreakAndThrow("Unhandled VertexColor reference mode in SceneImporter::ProcessVertex()\n");
			break;
		}

		FbxColor color = elementColor->GetDirectArray().GetAt(index);
		
		vertex.color = DirectX::XMFLOAT4(
			static_cast<float>(color.mRed),
			static_cast<float>(color.mGreen),
			static_cast<float>(color.mBlue),
			static_cast<float>(color.mAlpha)
		);
		DebugPrint("Color: %f %f %f %f\n", vertex.color.x, vertex.color.y, vertex.color.z, vertex.color.w);
	}

	// Retrieve the vertex normal
	for(int normalIndex = 0; normalIndex < numNormalsPerVertex; ++normalIndex)
	{
		FbxGeometryElementNormal* elementNormal = fbxMesh->GetElementNormal(normalIndex);
		int index;
		int directIndex;

		switch(elementNormal->GetMappingMode()) {
		case FbxGeometryElement::eByControlPoint:
			// DEBUG: This mode was not shown in the FBX sample.
			index = fbxVertexIndex;
			break;
		case FbxGeometryElement::eByPolygonVertex:
			index = totalVertexId;
			break;
		default:
			BreakAndThrow("Unhandled VertexColor mapping mode in SceneImporter::ProcessVertex()\n");
			break;
		}

		switch(elementNormal->GetReferenceMode()) {
		case FbxGeometryElement::eDirect:
			directIndex = index;
			break;
		case FbxGeometryElement::eIndexToDirect:
			directIndex = elementNormal->GetIndexArray().GetAt(index);
			break;
		default:
			BreakAndThrow("Unhandled VertexColor reference mode in SceneImporter::ProcessVertex()\n");
			break;
		}

		FbxVector4 normal = elementNormal->GetDirectArray().GetAt(directIndex);
		vertex.normal = FbxVectorToFloat3(normal);
	}

	// Add as a new vertex (the check for duplicates to reduce memory use isn't implemented yet)
	int newIndex = vertices.size();
	indices.push_back(newIndex);
	vertices.push_back(vertex);
}

shared_ptr<Scene> SceneImporter::InitCubeTestScene()
{
	// Create the scene that will be populated.
	auto resources = m_resources.lock();
	ASSERT(resources != nullptr);
	shared_ptr<Scene> scene = shared_ptr<Scene>(new Scene());
	scene->Initialize(resources);

	// Initial camera settings
	const float initialCamPos[] = { 0.0f, 0.0f, -20.0f };
	const float initialCamLookat[] = { 0.0f, 0.0f, 0.0f }; // Currently unused
	const float initialCamUp[] = { 0.0f, 1.0f, 0.0f }; // Currently unused

	// Add a camera
	auto camera = scene->AddCamera();
	camera->GetTransform()->SetLocalPosition(XMFLOAT3A(initialCamPos));

	auto emissiveMat = resources->m_matEmissive;

	// Add some cubes
	auto cube1  = scene->AddColorCube(XMFLOAT3A(  0, 0, 0));
	auto cube2  = scene->AddColorCube(XMFLOAT3A(  5, 0, 25));
	auto cube3  = scene->AddColorCube(XMFLOAT3A( -5, 0, 25));
	auto cube4  = scene->AddColorCube(XMFLOAT3A( 15, 0, 50));
	auto cube5  = scene->AddColorCube(XMFLOAT3A(-15, 0, 50));
	auto cube6  = scene->AddColorCube(XMFLOAT3A( 35, 0, 100));
	auto cube7  = scene->AddColorCube(XMFLOAT3A(-35, 0, 100));
	// Background bokeh cubes
	auto cube8  = scene->AddColorCube(XMFLOAT3A(-75, 50, 500));
	auto cube9  = scene->AddColorCube(XMFLOAT3A( 75, 50, 500));
	// Foreground bokeh cubes
	auto cube10 = scene->AddColorCube(XMFLOAT3A( 0.15f, -0.05f, -15));
	auto cube11 = scene->AddColorCube(XMFLOAT3A(-0.15f, -0.05f, -15));

	//cube1->m_transform->m_localScale = XMFLOAT3A(2, 2, 2);

	XMFLOAT3A cubeScale(2, 2, 2);
	cube2->GetTransform()->SetLocalScale(cubeScale);
	cube3->GetTransform()->SetLocalScale(cubeScale);
	cube4->GetTransform()->SetLocalScale(cubeScale);
	cube5->GetTransform()->SetLocalScale(cubeScale);
	cube6->GetTransform()->SetLocalScale(cubeScale);
	cube7->GetTransform()->SetLocalScale(cubeScale);

	XMFLOAT3A backgroundCubeSize(3, 3, 3);
	cube8->GetTransform()->SetLocalScale(backgroundCubeSize);
	cube9->GetTransform()->SetLocalScale(backgroundCubeSize);

	XMFLOAT3A foregroundCubeSize(0.025f, 0.025f, 0.025f);
	cube10->GetTransform()->SetLocalScale(foregroundCubeSize);
	cube11->GetTransform()->SetLocalScale(foregroundCubeSize);


	cube8->GetRenderer()->SetMaterial(emissiveMat);
	cube9->GetRenderer()->SetMaterial(emissiveMat);
	cube10->GetRenderer()->SetMaterial(emissiveMat);
	cube11->GetRenderer()->SetMaterial(emissiveMat);

	return scene;
}

shared_ptr<Scene> SceneImporter::InitLampsTestScene()
{
	// Create the scene that will be populated.
	auto resources = m_resources.lock();
	ASSERT(resources != nullptr);
	shared_ptr<Scene> scene = shared_ptr<Scene>(new Scene());
	scene->Initialize(resources);

	// Initial camera settings
	const float initialCamPos[] = { 0.0f, 2.0f, -20.0f };
	const float initialCamLookat[] = { 0.0f, 0.0f, 0.0f }; // Currently unused
	const float initialCamUp[] = { 0.0f, 1.0f, 0.0f }; // Currently unused

	// Add a camera
	auto camera = scene->AddCamera();
	camera->GetTransform()->SetLocalPosition(XMFLOAT3A(initialCamPos));

	auto emissiveMat = resources->m_matEmissive;
	auto cubeMesh = resources->m_meshCube;

	auto addLamp = [&](const XMFLOAT3A& position) -> shared_ptr<Entity> {
		// Create the object
		auto lamp = Entity::Create("Lamp");
		auto lamppost = Entity::Create("Lamppost");
		auto crossbar = Entity::Create("Crossbar");
		auto lampLight = Entity::Create("LampLight");

		// Parameters
		const float lamppostHeight = 3.0f;
		const float lamppostDiameter = 0.15f;
		const float crossbarLength = 1.0f;
		const float crossbarYOffset = 0.5f;
		const float lampBulbSize = 0.3f;

		// Use cube meshes to build lamps
		lamp->AddChild(lamppost);
		lamppost->AddMeshRenderer(cubeMesh);
		lamppost->GetTransform()->SetLocalScale(XMFLOAT3A(lamppostDiameter, lamppostHeight, lamppostDiameter));
		lamppost->GetTransform()->SetLocalPosition(XMFLOAT3A(0, lamppostHeight * 0.5f, 0));

		lamp->AddChild(crossbar);
		crossbar->AddMeshRenderer(cubeMesh);
		crossbar->GetTransform()->SetLocalScale(XMFLOAT3A(crossbarLength, lamppostDiameter * 0.5f, lamppostDiameter * 0.5f));
		crossbar->GetTransform()->SetLocalPosition(XMFLOAT3A(0, lamppostHeight - crossbarYOffset, 0));

		lamp->AddChild(lampLight);
		lampLight->AddMeshRenderer(cubeMesh);
		lampLight->GetTransform()->SetLocalScale(XMFLOAT3A(lampBulbSize, lampBulbSize, lampBulbSize));
		lampLight->GetTransform()->SetLocalPosition(XMFLOAT3A(0, lamppostHeight, 0));
		lampLight->GetRenderer()->SetMaterial(emissiveMat);

		// Set position
		lamp->GetTransform()->SetLocalPosition(position);

		scene->AddEntity(lamp);
		return lamp;
	};

	// Add two rows of lamps
	const int numLampsPerRow = 25;
	for(int i = 0; i < numLampsPerRow; ++i) {
		const float lampX = 3;
		const float lampDeltaZ = 25;
		addLamp(XMFLOAT3A( lampX, 0, -5 + i * lampDeltaZ));
		addLamp(XMFLOAT3A(-lampX, 0, -5 + i * lampDeltaZ));
	}

	// Color cube for flavor
	auto colorCube = scene->AddColorCube(XMFLOAT3A(0, 1, 0));
	XMVECTOR rotationVec = XMQuaternionRotationRollPitchYaw(M_PI_4, M_PI_4, 0);
	XMFLOAT4A rotation;
	XMStoreFloat4A(&rotation, rotationVec);
	colorCube->GetTransform()->SetLocalRotation(rotation);

	return scene;
}

shared_ptr<Scene> SceneImporter::InitHierarchyTestScene()
{
	// Create the scene that will be populated.
	auto resources = m_resources.lock();
	ASSERT(resources != nullptr);
	shared_ptr<Scene> scene = shared_ptr<Scene>(new Scene());
	scene->Initialize(resources);

	// Initial camera settings
	const float initialCamPos[] = { 0.0f, 0.0f, -20.0f };
	const float initialCamLookat[] = { 0.0f, 0.0f, 0.0f }; // Currently unused
	const float initialCamUp[] = { 0.0f, 1.0f, 0.0f }; // Currently unused

	// Add a camera
	auto camera = scene->AddCamera();
	camera->GetTransform()->SetLocalPosition(XMFLOAT3A(initialCamPos));

	auto emissiveMat = resources->m_matEmissive;
	auto cube = resources->m_meshCube;
	auto colorcube = resources->m_meshColorCube;

	// Create an "arm".
	// Each cube has its own base because the cubes are centered on the origin and we need to offset the arm rotations.
	auto upperArmBase = Entity::Create("Upper Arm Base");
	auto upperArm = Entity::Create("Upper Arm");
	auto foreArmBase = Entity::Create("Forearm Base");
	auto foreArm = Entity::Create("Forearm");

	float armRadius = 0.75f;
	float armLength = 2.0f;
	XMFLOAT3A armSize(armRadius, armLength, armRadius);
	XMFLOAT3A armOffset(0, armLength, 0);
	XMFLOAT3A cubeOffset(0, armLength * 0.5f, 0);

	XMFLOAT4A upperArmRotation;
	XMFLOAT4A foreArmRotation;
	XMStoreFloat4A(&upperArmRotation, XMQuaternionRotationRollPitchYaw(0, 0, M_PI_4));
	XMStoreFloat4A(&foreArmRotation, XMQuaternionRotationRollPitchYaw(0, 0, M_PI_4));

	upperArm->AddMeshRenderer(colorcube);
	foreArm->AddMeshRenderer(colorcube);

	// Set the scaling on the arm.
	upperArm->GetTransform()->SetLocalScale(armSize);
	foreArm->GetTransform()->SetLocalScale(armSize);

	// Set the rotations and offsets
	upperArmBase->GetTransform()->SetLocalRotation(upperArmRotation);
	foreArmBase->GetTransform()->SetLocalRotation(foreArmRotation);

	upperArm->GetTransform()->SetLocalPosition(cubeOffset);
	foreArmBase->GetTransform()->SetLocalPosition(armOffset);
	foreArm->GetTransform()->SetLocalPosition(cubeOffset);

	upperArmBase->AddChild(upperArm);
	upperArmBase->AddChild(foreArmBase);
	foreArmBase->AddChild(foreArm);

	scene->AddEntity(upperArmBase);

	return scene;
}

shared_ptr<Scene> SceneImporter::InitDepthTestScene()
{
	// Create the scene that will be populated.
	auto resources = m_resources.lock();
	ASSERT(resources != nullptr);
	shared_ptr<Scene> scene = shared_ptr<Scene>(new Scene());
	scene->Initialize(resources);

	// Initial camera settings
	const float initialCamPos[] = { 0.0f, 0.0f, 0.0f };

	// Add a camera
	auto camera = scene->AddCamera();
	camera->GetTransform()->SetLocalPosition(XMFLOAT3A(initialCamPos));

	auto colorCube0 = scene->AddColorCube(XMFLOAT3A(0, 0, 10));

	auto colorCube1 = scene->AddColorCube(XMFLOAT3A(0, 25, 250));
	colorCube1->GetTransform()->SetLocalScale(XMFLOAT3A(20, 20, 20));

	auto colorCube2 = scene->AddColorCube(XMFLOAT3A(50, 0, 500));
	colorCube2->GetTransform()->SetLocalScale(XMFLOAT3A(40, 40, 40));

	auto colorCube3 = scene->AddColorCube(XMFLOAT3A(0, -75, 750));
	colorCube3->GetTransform()->SetLocalScale(XMFLOAT3A(50, 50, 50));

	auto colorCube4 = scene->AddColorCube(XMFLOAT3A(-100, 0, 950));
	colorCube4->GetTransform()->SetLocalScale(XMFLOAT3A(50, 50, 50));

	return scene;
}

shared_ptr<Scene> SceneImporter::InitCubeGridScene()
{
	// Create the scene that will be populated.
	auto resources = m_resources.lock();
	ASSERT(resources != nullptr);
	shared_ptr<Scene> scene = shared_ptr<Scene>(new Scene());
	scene->Initialize(resources);

	// Initial camera settings
	const float initialCamPos[] = { 0.0f, 0.0f, 0.0f };

	// Add a camera
	auto camera = scene->AddCamera();
	camera->GetTransform()->SetLocalPosition(XMFLOAT3A(initialCamPos));

	// Add a grid of cubes
	int columns = 7;
	int rows = 5;
	float xMin = -10;
	float yMin = -6;

	float width = abs(xMin) * 2;
	float height = abs(yMin) * 2;

	// Hackfix for one row, one column
	if(columns <= 1) columns = 2;
	if(rows <= 1) rows = 2;

	float xDelta = width / (columns - 1);
	float yDelta = height / (rows - 1);

	float z = 40;

	// xIndex and yIndex are offsets from the center
	// e.g. 1 = 1 to the right of center
	for(int xIndex = 0; xIndex < columns; ++xIndex) {
		for(int yIndex = 0; yIndex < rows; ++yIndex) {
			float x = xMin + xIndex * xDelta;
			float y = yMin + yIndex * yDelta;

			auto colorCube = scene->AddColorCube(XMFLOAT3A(x, y, z));
		}
	}

	return scene;
}



}